﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;

using System.IO;
using DocumentFormat.OpenXml.Packaging;
using DocumentFormat.OpenXml.Spreadsheet;
using DocumentFormat.OpenXml;

using Framework;

namespace Framework.Tools.OpenXmlHelpers
{
    public class XlsxCell
    {
        private static readonly Regex _cellReferenceRegex = null;
        private static readonly Regex _rowIndexRegex = null;

        /// <summary>
        /// 静的コンストラクタ
        /// </summary>
        static XlsxCell()
        {
            _cellReferenceRegex = new Regex("^[A-Z]{1,3}[0-9]{1,7}$");
            _rowIndexRegex = new Regex("[0-9]{1,7}$");
        }

        private const int MaxColumnIndex = 16384;
        private const int MaxRowIndex = 1048576;

        public readonly bool Disable = false;
        public readonly string ColumnName = null;
        public readonly int ColumnIndex = 0;
        public readonly int RowIndex = 0;
        public readonly string CellReference = null;

        private readonly XlsxBook _book = null;
        private readonly XlsxSheet _sheet = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public XlsxCell(XlsxBook book, XlsxSheet sheet, string cellReference)
        {
            _book = book;
            _sheet = sheet;

            var cellRef = cellReference.Trim().ToUpper();

            if (_cellReferenceRegex.IsMatch(cellRef) == false)
            {
                //名前定義から取得
                var definedName = _book.DefinedNames(cellRef);
                if (definedName == null)
                {
                    this.Disable = true;
                    return;
                }
                var arr = definedName.Text.SplitBy('!');
                cellRef = arr[(arr.Length > 1 ? 1 : 0)].Replace("$", "");
                if (arr.Length > 1)
                {
                    _sheet = _book.Sheets(arr[0]);
                }
            }

            //セル参照から行番号を取得
            var match = _rowIndexRegex.Match(cellRef);
            var rowIndex = int.Parse(match.Value);
            if (rowIndex > MaxRowIndex)
            {
                this.Disable = true;
                return;
            }

            //セル参照から列名・列番号を取得
            var columnName = cellRef.Substring(0, match.Index);
            var columnIndex = GetColumnIndex(columnName);
            if (columnIndex < 1 || columnIndex > MaxColumnIndex)
            {
                this.Disable = true;
                return;
            }

            this.ColumnName = columnName;
            this.ColumnIndex = columnIndex;
            this.RowIndex = rowIndex;
            this.CellReference = columnName + rowIndex.ToString();
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public XlsxCell(XlsxBook book, XlsxSheet sheet, int columnIndex, int rowIndex)
        {
            _book = book;
            _sheet = sheet;

            if (columnIndex < 1 || columnIndex > MaxColumnIndex
                || rowIndex < 1 || rowIndex > MaxRowIndex)
            {
                this.Disable = true;
                return;
            }

            //列番号から列名取得
            var columnName = GetColumnName(columnIndex);

            this.ColumnName = columnName;
            this.ColumnIndex = columnIndex;
            this.RowIndex = rowIndex;
            this.CellReference = columnName + rowIndex.ToString();
        }

        /// <summary>
        /// Value
        /// </summary>
        public object Value
        {
            get
            {
                if (this.Disable == true) return null;

                var row = _sheet.Row(this.RowIndex);
                if (row == null) return null;

                var cell = _sheet.Cell(row, this.CellReference);
                if (cell == null) return null;

                var value = cell.InnerText;
                if (cell.DataType.Value == CellValues.SharedString)
                {
                    var idx = 0;
                    if (Int32.TryParse(cell.CellValue.Text, out idx) == true)
                    {
                        value = _book.SharedString(idx);
                    }
                }
                else if (cell.DataType.Value == CellValues.Boolean)
                {
                    switch (value)
                    {
                        case "0":
                            value = "FALSE";
                            break;
                        default:
                            value = "TRUE";
                            break;
                    }
                }

                return value;
            }
            set
            {
                if (this.Disable == true) return;
                if (value == null) return;

                var row = _sheet.Row(this.RowIndex);
                if (row == null) 
                {
                    row = _sheet.AddRow(this.RowIndex);
                }

                var cell = _sheet.Cell(row, this.CellReference);
                if (cell == null) 
                {
                    cell = _sheet.AddCell(row, this.CellReference);
                }

                var idx = _book.AddSharedString(value.ToString());

                cell.CellValue = new CellValue(idx.ToString());
                cell.DataType = new EnumValue<CellValues>(CellValues.SharedString);
            }
        }

        /// <summary>
        /// 現在のセルから指定分だけオフセットしたセルオブジェクトを返します。
        /// </summary>
        /// <param name="cx"></param>
        /// <param name="cy"></param>
        /// <returns></returns>
        public XlsxCell Offset(int cx, int cy)
        {
            return new XlsxCell(_book, _sheet, this.ColumnIndex + cx, this.RowIndex + cy);
        }

        /// <summary>
        /// 列名から列番号を取得します。
        /// </summary>
        internal static int GetColumnIndex(string columnName)
        {
            var len = columnName.Length;
            if (len == 1)
            {
                //１桁の場合
                return (columnName[0] - 64); 
            }
            else if (len == 2)
            {
                //２桁の場合
                return (columnName[1] - 64) * 26 + (columnName[0] - 64);
            }
            else
            {
                //３桁の場合
                return (columnName[2] - 64) * 676 + (columnName[1] - 64) * 26 + (columnName[0] - 64);
            }
        }

        /// <summary>
        /// 列番号から列名を取得します。
        /// </summary>
        internal static string GetColumnName(int columnIndex)
        {
            if (columnIndex <= 26)
            {
                //１桁の場合

                return new string(new[] { (char)(columnIndex + 64) });
            }
            else
            {
                var div1 = columnIndex / 26;
                var mod1 = columnIndex % 26;
                if (mod1 == 0)
                {
                    div1--;
                    mod1 = 26;
                }
                if (div1 <= 26)
                {
                    //２桁の場合
                    return new string(new[] { (char)(div1 + 64), (char)(mod1 + 64) });
                }
                else
                {
                    //３桁の場合

                    var div2 = div1 / 26;
                    var mod2 = div1 % 26;
                    if (mod2 == 0)
                    {
                        div2--;
                        mod2 = 26;
                    }
                    return new string(new[] { (char)(div2 + 64), (char)(mod2 + 64), (char)(mod1 + 64) });
                }
            }
        }
    }
}
